; Extending the capabilities of the model by importing relevant extensions.
extensions [csv rnd stats time]

; Breeding different types of turtles (agents) to represent various entities in the system.

; Entities responsible for building demolition.
breed [demolition_companies demolition_company]
; Entities responsible for building construction.
breed [construction_companies construction_company]
; Other types of building owners.
breed [other_owners other_owner]
; Suppliers for construction materials.
breed [material_suppliers material_supplier]
; Individual private owners of buildings.
breed [private_owners private_owner]
; Commercial company owners of buildings.
breed [company_owners company_owner]
; Housing corporations that own buildings.
breed [housing_corporation_owners housing_corporation_owner]

;----------------------------------------------------------------------------------------------------------------------




;----------------------------------------------------------------------------------------------------------------------
;Variables

globals [

  seed ; seed

  ; Material intensities for different types of buildings.
  mat_inten_mason ;For masonry-based buildings.
  mat_inten_wood ;For wood-based buildings.

  ; Weibull distribution parameters.
  weibull_para

  ; Rate at which materials are collected and recycled.
  coll_recy_rate

  ; Distribution of single vs. multi-family homes.
  opp_singe_multi

  ; Building distribution in Amsterdam by use type.
  distribution_build

  ; The starting year of the simulation.
  simulation_start_year
  ams_building_count_starting_year

  ; Subset of utility buildings in Amsterdam.
  utility_amsterdam

  ; Boolean to check if it's a new month.
  new_month

  ; Boolean to check if the outbox is empty.
  outbox_empty

  ; Quantity of various materials landfilled.
  concrete_landfilled
  wood_landfilled
  steel_landfilled

  ; Quantity of primary materials.
  primary_concrete
  primary_wood
  primary_steel

  ; Quantity of material requested
  concrete_requested
  wood_requested
  steel_requested

  ; Quantity of material demolished
  concrete_demolished
  wood_demolished
  steel_demolished


  ; Count of construction companies and housing associations.
  construction_company_count
  housing_association_count

  ; Distribution of buildings by ownership class.
  ownership


  ; Distribution of year and surface area for different types of homes.
  dist_year_surf_multi ;For multifamily homes.
  dist_year_surf_single ;For single-family homes.

  ; Number of buildings constructed or demolished during simulation.
  building_demolition_count
  building_construction_count

  ; Collection rates for demolition companies
  collection_rate_concrete
  collection_rate_wood
  collection_rate_steel


  ; Recycling rates for material suppliers
  recycling_rate_concrete
  recycling_rate_wood
  recycling_rate_steel

  ; Material cost different materials
  material_cost_concrete
  material_cost_wood
  material_cost_steel

  ; The radius in which owners check what kind of material type other building are.
  familiarity_radius_owners

  ; Reporters for analysis
  ratio_wood_based_buildings
  ratio_concrete_primary_requested
  ratio_wood_primary_requested
  ratio_steel_primary_requested
  ratio_concrete_landfilled_demolished
  ratio_wood_landfilled_demolished
  ratio_steel_landfilled_demolished

  ; Bias of owners for one material type (either wood-based or masonry-based construction).
  preference_bias

  ; Influence of specialization on the construction labour cost.
  specialization_weight

  ; Speed with wich a construction company adapts to its new environment.
  specialization_adaption_speed

  ; The basic labour cost per m2
  labour_cost_per_m2

  ; The capacities for various material of the material supplier
  capacity_wood_stock_material_supplier
  capacity_concrete_stock_material_supplier
  capacity_steel_stock_material_supplier

  ;Construction learning
  learning_speed
  learning_coefficient

  ;Construction time definitions
  construction_time_wood_based
  construction_time_masonry_based

  ; Project capacity of construction compane
  project_capacity_construction_companies

  ratio_wood_specilization_cc_0
  ratio_wood_specilization_cc_1
  ratio_wood_specilization_cc_2

  wood_based_specialization


  ; Time parameter for the simulation.
  dt
]

; Attributes specific to demolition companies.

demolition_companies-own[
  inbox ; Messages or notifications received.
  outbox ; Messages or notifications sent.
  projects ; Demolition projects being handled.
  ; Stocks of various materials after demolition.
  concrete_stock
  wood_stock
  steel_stock
  multiple_links_owners ; Links to building owners for various projects.
]

; Attributes specific to construction companies.
construction_companies-own[
  inbox
  outbox
  projects ; Construction projects being handled.
  specialization ; Whether the company specializes in wood or concrete buildings.
  multiple_links_owners ; Links to clients or building owners.
  project_capacity ; Total number of projects the company can handle.
  remaining_project_capacity ; Projects slots that are still available.
]

; Attributes specific to material suppliers.
material_suppliers-own[
  inbox
  outbox
  ; Stocks of various construction materials.
  wood_stock
  concrete_stock
  steel_stock
]

; Attributes for other building owners.
other_owners-own[
  inbox
  outbox
  assets ; Buildings owned by the entity.
]

; Attributes for private building owners.
private_owners-own[
  inbox
  outbox
  assets ; Buildings owned by the individual.
]

; Attributes for company building owners.
company_owners-own[
  inbox
  outbox
  assets ; Buildings owned by the company.
]

; Attributes for housing corporation owners.
housing_corporation_owners-own[
  inbox
  outbox
  assets ; Buildings owned by the housing corporation.
]

; Attributes for each patch (representing a piece of land or building).
patches-own[
  ; Current status of the patch.
  status ; (e.g., occupied, empty, under construction, etc.)

  ; Boolean to determine if a patch is part of the built environment.
  part_of_built_environment

  ; Time duration the patch has been empty.
  time_empty

  ; Building-specific attributes if a building is present on the patch.
  building_archetype ; Type of the building (e.g., residential, commercial).
  building_construction_year ; Year when the building was constructed.
  building_floor_surface ; Total floor area of the building.

  ; The expected remaining duration before the building needs replacement or major renovation.
  remaining_lifespan

  ; Who owns the building.
  owner

  ; Material composition of the building.
  kg_concrete
  kg_wood
  kg_steel

  ; Type of primary material used in the building.
  material_type ; (e.g., masonry-based or wood-based).
]


;----------------------------------------------------------------------------------------------------------------------
;Setup

to setup

if not sensitivity_analysis [
  clear-all                          ; Clears everything from the world, like turtles, patches, links, etc.
  reset-ticks                        ; Resets the tick counter to zero
  clear-output                       ; Clears the output area
  ]


  set ams_building_count_starting_year 498829 ; Initial number of buildings in Amsterdam's built environment in the real world.
  set simulation_start_year 2021 ; The year in which the simulation starts.
  set dt time:create "2021/01/01" ; Exact start date for the simulation.
  set utility_amsterdam csv:from-file "data_amsterdam_utility.csv" ; CSV file with a subset of buildings
  set distribution_build [["apartment" 395609] ["single_family" 4892] ["row" 49488] ["office" 9739] ["other" 39098]] ; Number of buildings of each type in Amsterdam (2021).
  set mat_inten_mason [["apartment" 19 883 31] ["single_family" 216 974 34] ["row" 39 353 12] ["office" 22 615 22] ["other" 35 564 20]] ; Material intensities for buildings with masonry, categorized by wood, concrete, and steel.
  set mat_inten_wood [["apartment" 144 157 8] ["single_family" 75 677 19] ["row" 150 374 41] ["office" 200 272 16] ["other" 142 370 21]]; Material intensities for buildings with wood, categorized by wood, concrete, and steel.
  set weibull_para [2.95 130] ; Parameters for Weibull distribution for building lifespan estimation.



; If a sensitivity analysis is executed, the parameters are defined in the Python script.
if not sensitivity_analysis [
    random-seed 12345
    set capacity_wood_stock_material_supplier 15000
    set capacity_concrete_stock_material_supplier 50000
    set capacity_steel_stock_material_supplier 15000
    set labour_cost_per_m2 922
    set project_capacity_construction_companies 500
    set preference_bias 0.1 ;bias component for owners to determine their preference.
    set material_cost_concrete 0.51
    set material_cost_wood 1.28
    set material_cost_steel 1.2
    set collection_rate_concrete 0.85
    set collection_rate_wood 0.95
    set collection_rate_steel 0.95
    set recycling_rate_concrete 0.02
    set recycling_rate_wood 0.32
    set recycling_rate_steel 0.85
    set learning_speed 1
    set wood_based_specialization 0
  ]


  set familiarity_radius_owners 2
  set opp_singe_multi [113 71] ; Number of single-family and multi-family homes in Amsterdam (2021).
  set ownership [["private_owner" 269139] ["housing_corporation_owner" 179630]] ; Number of buildings per ownership type in Amsterdam (2021).
  set dist_year_surf_multi [[1000 1850 77 19632] [1850 1905 71 39347][1905 1925 69 52980][1925 1945 68 63394][1945 1955 57 12904][1955 1965 65 25236][1965 1975 74 26926][1975 1985 69 35851][1985 1995 70 34641][1995 2005 88 24988][2005 2015 87 32797][2015 2021 58 26913]] ;The amount of multi-family homes that were built between the specified time interval together with their respective floor surface in Amsterdam (2021).
  set dist_year_surf_single [[1000 1850 171 1823] [1850 1905 209 1106][1905 1925 100 3616][1925 1945 98 5343][1945 1955 81 3106][1955 1965 96 6341][1965 1975 114 2326][1975 1985 98 3649][1985 1995 98 10469][1995 2005 127 7176][2005 2015 137 5853][2015 2021 144 3571]] ;The amount of single-family homes that were built between the specified time interval together with their respective floor surface in Amsterdam (2021).
  set learning_coefficient 0.9

  ;Counters
  ; Setting counters for tracking the number of buildings constructed and demolished during the simulation
  set building_demolition_count 0
  set building_construction_count 0
  set construction_time_wood_based 12
  set construction_time_masonry_based 3
  ; Setting the number of specific entities in the simulation
  set construction_company_count 3   ; Number of construction companies in the simulation.
  set housing_association_count 3    ; Number of housing associations in the simulation.

  setup-built-environment            ; Call the procedure to setup the built environment in the simulation.

end


; Update reporters
to update_reporters
  carefully [set ratio_wood_based_buildings count patches with [material_type = "wood_type"] / (count patches with [status = "occupied"])][set ratio_wood_based_buildings 0]
  carefully [set ratio_concrete_primary_requested primary_concrete / concrete_requested][set ratio_concrete_primary_requested 0]
  carefully [set ratio_wood_primary_requested primary_wood / wood_requested][set ratio_wood_primary_requested 0]
  carefully [set ratio_steel_primary_requested primary_steel / steel_requested][set ratio_steel_primary_requested 0]
  carefully [set ratio_concrete_landfilled_demolished concrete_landfilled / concrete_demolished][set ratio_concrete_landfilled_demolished 0]
  carefully [set ratio_wood_landfilled_demolished wood_landfilled / wood_demolished][set ratio_wood_landfilled_demolished 0]
  carefully [set ratio_steel_landfilled_demolished steel_landfilled / steel_demolished][set ratio_steel_landfilled_demolished 0]
  set ratio_wood_specilization_cc_0  [item 0 specialization] of construction_company 0 / 20
  set ratio_wood_specilization_cc_1  [item 0 specialization] of construction_company 1 / 20
  set ratio_wood_specilization_cc_2  [item 0 specialization] of construction_company 2 / 20

end


; Reset various indicators to 0
to reset_indicators
  set primary_concrete 0
  set primary_wood 0
  set primary_steel 0
  set concrete_requested 0
  set wood_requested 0
  set steel_requested 0
  set concrete_landfilled 0
  set wood_landfilled 0
  set steel_landfilled 0
  set concrete_demolished 0
  set wood_demolished 0
  set steel_demolished 0
end


to setup-built-environment

  ; Mark inner patches (excluding borders) as part of the built environment.
  ask patches [
    if pxcor > 1 and pxcor < max-pxcor - 1 and pycor > 1 and pycor < max-pycor - 1 [
      set part_of_built_environment 1
    ]
  ]

  ; Initialize construction companies.
  ; Place them vertically on the left side.
  let create_patch_margin_count max-pycor - 2
  repeat construction_company_count [
    create-construction_companies 1 [
      setxy 0 create_patch_margin_count
      set inbox []
      set outbox []
      set projects []
      set color white
      set shape "i beam"
      set specialization (list wood_based_specialization (20 - wood_based_specialization))
      set multiple_links_owners []
      set project_capacity project_capacity_construction_companies
    ]
    ; Adjust y-coordinate for the next construction company.
    if construction_company_count > 1 [set create_patch_margin_count create_patch_margin_count - (int (max-pycor - 4) / (construction_company_count - 1))]
  ]

  ; Initialize housing associations.
  ; Place them horizontally on the bottom.
  set create_patch_margin_count max-pxcor - 2
  repeat housing_association_count [
    create-housing_corporation_owners 1 [
      setxy create_patch_margin_count 0
      set assets []
      set color white
      set inbox []
      set outbox []
      set shape "orbit 5"
    ]
    ; Adjust x-coordinate for the next housing association.
    if housing_association_count > 1 [set create_patch_margin_count create_patch_margin_count - (int (max-pxcor - 4) / (housing_association_count - 1))]
  ]

  ; Initialize a single demolition company.
  ; Position it centrally on the right side.
  create-demolition_companies 1 [
    setxy max-pycor  max-pycor / 2
    set concrete_stock 0
    set wood_stock 0
    set steel_stock 0
    set inbox []
    set outbox []
    set projects []
    set color white
    set shape "x"
    set multiple_links_owners []
  ]

  ; Initialize a material supplier.
  ; Position it centrally on the top side.
  create-material_suppliers 1 [
    setxy max-pxcor / 2 max-pycor
    set wood_stock 0
    set concrete_stock 0
    set steel_stock 0
    set inbox []
    set outbox []
    set shape "logs"
    set color white
  ]

  ; Define the occupancy of patches based on municipal land ratio.
  ; Some patches are occupied by buildings, others are municipal land.
  ask patches with [part_of_built_environment = 1][
      set material_type "masonry_type"
      set building_archetype first rnd:weighted-one-of-list distribution_build [ [p] -> last p ] ; Assign building type based on Amsterdam's composition.
      assign_owner ; Assign owner to each building patch.
      set status "occupied"
      set pcolor black
  ]

  ; Initialize assets for the environment.
  assign_assets_environment_setup
end
; Function to assign parameters to every patch of the built environment.
to assign_assets_environment_setup
  ; Map each distribution build to its first element and store it in building_archetypes_map.
  let building_archetypes_map map [x -> first x] distribution_build

  ; For each building type, assign attributes based on its archetype.
  foreach building_archetypes_map [
    building_archetype_func ->

    ; Assign attributes for apartment buildings.
    if building_archetype_func = "apartment" [
      ask patches with [building_archetype = "apartment"][
        ; Randomly select a construction year time interval based on the distribution observed in Amsterdam.
        let temp_dist_year_surf_multi rnd:weighted-one-of-list dist_year_surf_multi [ [p] -> last p ]
        ; Randomly choose a construction year within the selected time interval.
        set building_construction_year random (item 1 temp_dist_year_surf_multi - item 0 temp_dist_year_surf_multi) + item 0 temp_dist_year_surf_multi
        ; Allocate the floor surface based on the selected time interval.
        set building_floor_surface item 2 temp_dist_year_surf_multi
        ; Compute the remaining lifespan based on the selected time interval.
        set remaining_lifespan calculate_remaining_lifespan building_construction_year
        ; Compute the quantity of materials based on the building archetype and surface.
        let material_quantity_values calculate_material_quantity "masonry_type" building_archetype_func building_floor_surface
        set kg_concrete item 0 material_quantity_values
        set kg_wood item 1 material_quantity_values
        set kg_steel item 2 material_quantity_values
      ]
    ]

    ; Assign attributes for single family and row houses.
    if member? building_archetype_func ["single_familiy" "row"][
      ask patches with [member? building_archetype ["single_family" "row"]][
        ; Randomly select a time interval for the construction year based on the distribution observed in Amsterdam.
        let dist_year_surf_single_func rnd:weighted-one-of-list dist_year_surf_single [ [p] -> last p ]
        ; Randomly choose a construction year within the selected time interval.
        set building_construction_year random (item 1 dist_year_surf_single_func - item 0 dist_year_surf_single_func) + item 0 dist_year_surf_single_func
        ; Allocate the floor surface based on the selected time interval.
        set building_floor_surface item 2 dist_year_surf_single_func
        ; Compute the remaining lifespan based on the selected time interval.
        set remaining_lifespan calculate_remaining_lifespan building_construction_year
        ; Compute the quantity of materials based on the building archetype and surface.
        let material_quantity_values calculate_material_quantity "masonry_type" building_archetype_func building_floor_surface
        set kg_concrete item 0 material_quantity_values
        set kg_wood item 1 material_quantity_values
        set kg_steel item 2 material_quantity_values
      ]
    ]

    ; Assign attributes for utility buildings like office, shop, education etc.
    if member? building_archetype_func ["office" "shop" "education" "health_care" "meetings" "recreation" "industry" "sports" "other"] [
      ; Filter utility buildings based on the building archetype function.
      let utility_list_building_func filter [x -> item 3 x = building_archetype_func] utility_amsterdam
      ask patches with [member? building_archetype ["office" "shop" "education" "health_care" "meetings" "recreation" "industry" "sports" "other"]][
        ; Randomly select the surface and construction year time for a building based on the distribution observed in Amsterdam.
        let surface_construction_year_values select_from_data utility_list_building_func
        set building_floor_surface item 0 surface_construction_year_values
        set building_construction_year item 1 surface_construction_year_values
        ; Compute the remaining lifespan based on the Weibull distribution.
        set remaining_lifespan calculate_remaining_lifespan building_construction_year
        ; Compute the quantity of materials based on the building archetype and surface.
        let material_quantity_values calculate_material_quantity "masonry_type" building_archetype_func building_floor_surface
        set kg_concrete item 0 material_quantity_values
        set kg_wood item 1 material_quantity_values
        set kg_steel item 2 material_quantity_values
      ]
    ]
  ]
end

; Function to randomly select a building from a subset of utility buildings.
to-report select_from_data[utility_buildings]
  let random_utility_buildings_index random (length utility_buildings) ; Get a random index from the utility buildings list.
  ; Extract floor surface and construction year based on the random index.
  let floor_surface_func item 1 item random_utility_buildings_index utility_buildings
  let construction_year_func item 2 item random_utility_buildings_index utility_buildings
  ; Report back a list containing floor surface and construction year.
  report (list floor_surface_func construction_year_func)
end



to assign_owner
  ; Based on the type of building, this function will decide which entity should own it.

  ; Check if the building is one of the residential types.
  ifelse building_archetype = "apartment" or building_archetype = "single_family" or building_archetype = "row"[

    ; Decide the type of owner based on a distribution.
    let owner_breed first rnd:weighted-one-of-list ownership [ [p] -> last p ]

    ; Check if the building is owned by a private owner.
    ifelse owner_breed = "private_owner" [

      ; If so, create a new private owner agent.
      sprout-private_owners 1 [
        set assets []
        set assets lput myself assets
        set outbox []
        set inbox []
        set shape "dot"
      ]
      set owner "private"

    ][
      ; Check if the building is owned by a housing corporation.
      ifelse owner_breed = "housing_corporation_owner" [

        ; If so, assign the building to one of the existing housing corporation agents.
        ask one-of housing_corporation_owners[
          set assets lput myself assets
        ]
        set owner "housing corporation"
      ][
        ; For all other cases, create an "private" agent.
        sprout-other_owners 1[
          set assets []
          set assets lput myself assets
          set outbox []
          set inbox []
          set shape "triangle"
        ]
        set owner "other owner"
      ]
    ]

  ][
    ; For all non-residential buildings, allocate to corporations.
    sprout-company_owners 1[
      set assets []
      set assets lput myself assets
      set outbox []
      set inbox []
      set shape "triangle 2"
    ]
    set owner "private"
  ]
end

;---------------- LIFESPAN CALCULATIONS ------------------

to-report calculate_remaining_lifespan[construction_year]
  ; Calculates how many months are remaining in the lifespan of a building based on its construction year.

  let lifespan 0
  let remaining_lifespan_func "INF"

  ; If the building was constructed after 1900, calculate its lifespan.
  if construction_year > 1900 [
    set lifespan calculate_weilbull

    ; Check if the building would have been demolished by the start of the simulation.
    while [construction_year + lifespan < simulation_start_year][
      set lifespan calculate_weilbull
    ]
  ]

  ; Return the number of months remaining in the building's lifespan.
  report int ((construction_year + lifespan - 2021) * 12)
end

to-report calculate_weilbull
  ; Calculates the lifespan of a building using the Weibull distribution.

  let λ item 1 weibull_para
  let k item 0 weibull_para

  ; Return a lifespan value generated from the Weibull distribution.
  report  λ * (-1 * ln(1 - random-float 1 ))^ (1 / k )
end

;---------------- MATERIAL QUANTITY CALCULATIONS ------------------

to-report calculate_material_quantity[material_type_func building_archetype_func building_floor_surface_func]
  ; Calculates the quantity of material (concrete, wood, steel) used based on building type and material type.

  let kg_concrete_func 0
  let kg_wood_func 0
  let kg_steel_func 0

  ; Check if the building uses masonry type materials.
  ifelse material_type_func = "masonry_type" [

    ; Get the list of building archetypes for masonry.
    let building_archetypes_mat_inten_mason map [x -> first x] mat_inten_mason

    ; Default to "other" unless a match is found.
    let item_position_building_archetype position "other" building_archetypes_mat_inten_mason
    let building_index position building_archetype_func building_archetypes_mat_inten_mason

    ; If a match is found, update the position.
    if is-number? building_index [
      set item_position_building_archetype building_index
    ]

    ; Calculate the quantity of each material for the building.
    set kg_concrete_func int item 2 item item_position_building_archetype mat_inten_mason * building_floor_surface_func
    set kg_wood_func int item 1 item item_position_building_archetype mat_inten_mason * building_floor_surface_func
    set kg_steel_func int item 3 item item_position_building_archetype mat_inten_mason * building_floor_surface_func

  ][
    ; Else, calculate material values for buildings using wood type materials.

    ; Get the list of building archetypes for wood.
    let building_archetypes_mat_inten_wood map [x -> first x] mat_inten_wood

    ; Default to "other" unless a match is found.
    let item_position_building_archetype position "other" building_archetypes_mat_inten_wood
    let building_index position building_archetype_func building_archetypes_mat_inten_wood

    ; If a match is found, update the position.
    if is-number? building_index [
      set item_position_building_archetype building_index
    ]

    ; Calculate the quantity of each material for the building.
    set kg_concrete_func int item 2 item item_position_building_archetype mat_inten_wood * building_floor_surface_func
    set kg_wood_func int item 1 item item_position_building_archetype mat_inten_wood * building_floor_surface_func
    set kg_steel_func int item 3 item item_position_building_archetype mat_inten_wood * building_floor_surface_func

  ]

  ; Return the quantities of concrete, wood, and steel.
  report (list kg_concrete_func kg_wood_func kg_steel_func)
end

;----------------------------------------------------------------------------------------------------------------------
; 'go' Procedure
to go
  ;reset_indicators

  ; Initialization
  set new_month TRUE          ; Indicate start of a new month
  set outbox_empty FALSE      ; Indicate that the outbox is not empty yet
  let while_counter 0         ; Initialize the while loop counter

  if show_new_tick [
  show "---------------------------------------------------"
  show ticks
  ]

  ; This while loop runs as long as the outbox is not empty
  while [NOT outbox_empty] [

    ; Ask private, company, other and housing corporation owners to evaluate their building stock
    ask turtles with [breed = private_owners or breed = company_owners or breed = other_owners or breed = housing_corporation_owners] [
      evaluate_building_stock
    ]

    ; Ask the same owners to handle their inbox messages
    ask turtles with [breed = private_owners or breed = company_owners or breed = other_owners or breed = housing_corporation_owners] [
      handle_inbox_owners
    ]

    ; Ask demolition companies to handle their inbox messages
    ask demolition_companies [
      handle_inbox_demolition_companies
    ]

    ; Ask demolition companies to check their demolition projects and clear their demolition waste stock
    ask demolition_companies [
      check_demolition_projects
      clear_demolition_waste_stock
    ]

    ; Ask material suppliers to handle their inbox messages
    ask material_suppliers [
      handle_inbox_material_suppliers
    ]


    ; Ask construction companies to handle their inbox messages
    ask construction_companies [
      handle_inbox_construction_company
    ]

    ; Ask construction companies to check their construction projects
    ask construction_companies [
      check_construction_projects
    ]


    ; Set the new month to FALSE indicating operations for the new month are done
    set new_month FALSE

    ; Initially set outbox_empty to TRUE, this will be checked and set to FALSE if any turtle has a non-empty outbox
    set outbox_empty TRUE
    ask turtles [
      if not empty? outbox [set outbox_empty FALSE
        stop
      ]
    ]

    ; Ask all turtles to send their messages present in their outbox
    ask turtles [
      send_messages_outbox
    ]


  ]

  ; Monitor patches that are occupied. Reduce their lifespan by one month.
  ask patches with [status = "occupied" AND is-number? remaining_lifespan] [
    set remaining_lifespan remaining_lifespan - 1
  ]

  ; Monitor patches that are empty. Increase their empty time by one month.
  ask patches with [status = "empty"] [
    set time_empty time_empty + 1
  ]


  update_reporters ; Update reporters for analysis

  ; Add one month to the display time
  set dt time:plus dt 1 "months"
  tick ; Advance the simulation time by one step


end

;--------------------------
; Construction Companies
;--------------------------


; This function handles messages related to construction projects. It can calculate construction costs based on different building types or
; can add new construction projects to the queue.
to handle_inbox_construction_company

  ; Check if the inbox is empty. If yes, then stop the function.
  if empty? inbox [stop]

  ; Map to get the first item from the construction cost building archetypes list.
  ;let construction_cost_masonry_map map [x -> first x] construction_cost_building_archetypes

  ; Count the number of messages in the inbox.
  let message_count_inbox length inbox

  ; Loop through all messages in the inbox.
  repeat message_count_inbox [

    ; Get the first message from the inbox.
    let message receive_first_message

    ; Handle messages that request construction costs.
    if last message = "construction_cost_request" [

      ; Optional action display.
      if show_actions [show "Construction company - Handle construction_cost_request"]


      ; Extract relevant data from the message.
      let building_patch_func item 0 message
      let owning_agent_func item 1 message
      let building_archetype_func item 2 message
      let building_floor_surface_func item 3 message

      ; Update the status of the building patch.
      ask building_patch_func [set status "waiting_for_material_request"]

      ; Calculate material quantity required for different building types.
      let material_quantity_list_wood_type calculate_material_quantity "wood_type" building_archetype_func building_floor_surface_func
      let material_quantity_list_masonry_type calculate_material_quantity "masonry_type" building_archetype_func building_floor_surface_func

      ; Create messages for requesting material cost for different types.
      let material_cost_wood_type_message (list material_supplier 7 building_patch_func self owning_agent_func building_archetype_func building_floor_surface_func "wood_type" item 0 material_quantity_list_wood_type item 1 material_quantity_list_wood_type item 2 material_quantity_list_wood_type "primary_material_cost_request")
      let material_cost_masonry_type_message (list material_supplier 7 building_patch_func self owning_agent_func building_archetype_func building_floor_surface_func "masonry_type" item 0 material_quantity_list_masonry_type item 1 material_quantity_list_masonry_type item 2 material_quantity_list_masonry_type "primary_material_cost_request")

      ; Add the created messages to the outbox.
      set outbox lput material_cost_wood_type_message outbox
      set outbox lput material_cost_masonry_type_message outbox

    ]

    ; Handle messages related to construction commissions.
    if last message = "construction_commission" [



      ; Update any pending project's capacity.
      update_remaining_project_capacity

      ; Extract relevant data from the message.
      let building_patch_func item 0 message
      let owning_agent_func item 1 message
      let material_type_func item 2 message
      let building_archetype_func item 3 message
      let building_floor_surface_func item 4 message

      ; Check if there's enough capacity to handle the new project.
      ifelse remaining_project_capacity >= building_floor_surface_func [


        ; Optional action display.
        if show_actions [show "Construction company - Handle construction_commission / start construction"]

        ifelse material_type = "wood_type" [
          let material_quantity_list_wood_type calculate_material_quantity "wood_type" building_archetype_func building_floor_surface_func
          let material_cost_wood_type_message (list material_supplier 7 self item 0 material_quantity_list_wood_type item 1 material_quantity_list_wood_type item 2 material_quantity_list_wood_type "primary_material_commission")
          set outbox lput material_cost_wood_type_message outbox
        ][
          let material_quantity_list_masonry_type calculate_material_quantity "masonry_type" building_archetype_func building_floor_surface_func
          let material_cost_masonry_type_message (list material_supplier 7 self item 0 material_quantity_list_masonry_type item 1 material_quantity_list_masonry_type item 2 material_quantity_list_masonry_type "primary_material_commission")
          set outbox lput material_cost_masonry_type_message outbox
        ]


        ; Default construction time.
        let construction_time construction_time_masonry_based

        ; If the material type is masonry, then set a different construction time.
        if material_type = "masonry_type" [set construction_time construction_time_wood_based]

        ; Update the status of the building patch.
        ask building_patch_func [set status "under_construction"]

        ; Link the owning agent if not already linked.
        if member? owning_agent_func link-neighbors [
          set multiple_links_owners lput owning_agent_func multiple_links_owners
        ]
        create-link-with owning_agent_func [
          set shape "dashed"
          set thickness 0.15
        ]

        ; Add the new project to the projects list.
        let project (list building_patch_func owning_agent_func material_type_func building_archetype_func building_floor_surface_func construction_time)
        set projects lput project projects

      ][
        ; If not enough capacity, update the building patch status to empty.
        ask building_patch_func [
          set status "empty"
          set pcolor white
        ]
      ]
    ]

    ; Handle material response messages for construction projects.
    if last message = "primary_material_response" [

      ; Optional action display.
      if show_actions [show "Construction company - Handle primary material response / send build cost estimate"]

      ; Extract relevant data from the message.
      let building_patch_func item 0 message
      let material_agent_func item 1 message
      let owning_agent_func item 2 message
      let building_archetype_func item 3 message
      let building_floor_surface_func item 4 message
      let material_type_func item 5 message
      let material_cost_func item 6 message

      ; Calculate construction cost and send a cost estimate message based on the material type.
      ifelse material_type_func = "wood_type" [

        ; Calculate cost for wood type buildings.
        let construction_cost_estimate_wood_type calculate_construction_cost "wood_type" building_archetype_func building_floor_surface_func material_cost_func

        ; Create a message with the calculated cost estimate.
        let construction_cost_estimate_wood_type_message (list owning_agent_func building_patch_func self "wood_type" building_archetype_func building_floor_surface_func construction_cost_estimate_wood_type "construction_cost_estimate")



        ; Add the message to the outbox.
        set outbox lput construction_cost_estimate_wood_type_message outbox

      ][


        ; Calculate cost for masonry type buildings.
        let construction_cost_estimate_masonry_type calculate_construction_cost "masonry_type" building_archetype_func building_floor_surface_func material_cost_func



        ; Create a message with the calculated cost estimate.
        let construction_cost_estimate_masonry_type_message (list owning_agent_func building_patch_func self "masonry_type" building_archetype_func building_floor_surface_func construction_cost_estimate_masonry_type "construction_cost_estimate")

        ; Add the message to the outbox.
        set outbox lput construction_cost_estimate_masonry_type_message outbox

      ]
    ]
  ]
end


; This function constructs a building and assigns the relevant building parameters.
to construct_building[building_patch material_type_func building_archetype_func building_floor_surface_func]

    ; Calculate the material quantities needed for the construction
    let material_quantities calculate_material_quantity material_type_func building_archetype_func building_floor_surface_func

    ; Get the current year from system date
    let current_year read-from-string time:show dt "yyyy"
    let remaining_lifespan_func calculate_remaining_lifespan current_year

    ; Update attributes of the specific building patch
    ask building_patch [
      set status "occupied"
      set pcolor black

      ; Optionally show construction actions, if enabled
      if show_actions [show "Construction company - construct building"]

      set time_empty 0
      ; Set building attributes based on given parameters
      set material_type material_type_func
      set building_archetype building_archetype_func
      set building_floor_surface building_floor_surface_func
      ; Assign calculated material quantities to specific material types
      set kg_concrete item 0 material_quantities
      set kg_steel item 1 material_quantities
      set kg_wood item 2 material_quantities
      set building_construction_year current_year
      set remaining_lifespan remaining_lifespan_func
    ]

    ; Adjust specialization based on material type
    specialization_adapter  material_type_func
    ; Update the total count of constructed buildings
    set building_construction_count building_construction_count + 1
end

; This function updates the project capacity for construction companies.
to update_remaining_project_capacity
  ask construction_companies [
    ; Calculate and set remaining project capacity for each construction company
    let project_capacity_list map [project -> item 4 project] projects
    let project_capacity_sum sum project_capacity_list
    set remaining_project_capacity project_capacity - project_capacity_sum
  ]
end

; This function calculates the construction cost for buildings based on their attributes and material costs.
to-report calculate_construction_cost [material_type_func building_archetype_func building_floor_surface_func material_cost]


  ; Calculate construction cost based on material type
  ifelse material_type_func = "masonry_type" [
    ; calculate cost efficient with Wright's law learning curve effect
    let cost_coefficient calculate_cost_coefficient item 1 specialization

    ; Calculate the construction cost for masonry type buildings
    let construction_cost_masonry_type building_floor_surface_func * labour_cost_per_m2 * cost_coefficient + material_cost
    report construction_cost_masonry_type
  ][
    ; calculate cost efficient with Wright's law learning curve effect
    let cost_coefficient calculate_cost_coefficient item 0 specialization


    ; Calculate the construction cost for wood type buildings
    let construction_cost_wood_type building_floor_surface_func * labour_cost_per_m2 * cost_coefficient + material_cost
    report construction_cost_wood_type
  ]
end


to-report calculate_cost_coefficient [number_of_buildings]
  let factor 1
  if number_of_buildings > 0 [
    let exponent (log 0.9 2)
    set factor (number_of_buildings ^ exponent)
  ]
  report factor
end


; This function checks all active projects of a construction company.
to check_construction_projects
  ; List to store projects after being checked
  let projects_checked []

  ; Stop if not a new month or no projects available
  if not new_month [stop]
  if empty? projects [stop]

  ; Check each project in the projects list
  foreach projects [
    project ->
    ; Extract project attributes
    let building_patch_func item 0 project
    let owning_agent_func item 1 project
    let material_type_func item 2 project
    let building_archetype_func item 3 project
    let building_floor_surface_func item 4 project
    let construction_time item 5 project

    ; Check construction time for the project
    ifelse (construction_time) = 0 [
      ; If construction time is zero, construct the building.
      construct_building building_patch_func material_type_func building_archetype_func building_floor_surface_func

      ; Handle links with owning agents
      ifelse (position owning_agent_func multiple_links_owners) = FALSE [
        ask link-with owning_agent_func [die] ; remove the link if not found in multiple_links_owners
      ][
        ; Remove owning agent from multiple_links_owners list if it exists
        set multiple_links_owners remove (position owning_agent_func multiple_links_owners) multiple_links_owners
      ]
    ]
    [
      ; Decrease construction time by one month if it's more than zero
      let construction_time_updated construction_time - 1
      let project_checked (list building_patch_func owning_agent_func material_type_func building_archetype_func building_floor_surface_func construction_time_updated)
      set projects_checked lput project_checked projects_checked
    ]
  ]
  ; Update the projects list with the checked projects
  set projects projects_checked
end

; This function updates the specialization parameters based on the material type.
to specialization_adapter [material_type_func]
  ; Define lower and upper bounds for specialization
  let lower_bound 0
  let upper_bound 20  ; adjust as needed based on your model's logic

  ; Check if the given material type is "wood_type"
  ifelse material_type_func = "wood_type" [
    ; Increase "wood_type" specialization if below upper bound
    if item 0 specialization < upper_bound [
      set specialization replace-item 0 specialization (item 0 specialization + learning_speed)
    ]
    ; Decrease other specialization if above lower bound
    if item 1 specialization > lower_bound [
      set specialization replace-item 1 specialization (item 1 specialization - learning_speed)
    ]
  ][
    ; Otherwise, adjust for other material type
    ; Increase other specialization if below upper bound
    if item 1 specialization < upper_bound [
      set specialization replace-item 1 specialization (item 1 specialization + learning_speed)
    ]
    ; Decrease "wood_type" specialization if above lower bound
    if item 0 specialization > lower_bound [
      set specialization replace-item 0 specialization (item 0 specialization - learning_speed)
    ]
  ]
end

;--------------------------
; Demolition Companies
;--------------------------

; This function processes all messages in the inbox for the demolition companies.
to handle_inbox_demolition_companies
  ; Exit if there's no message in the inbox
  if empty? inbox [stop]

  let message_count_inbox length inbox

  ; Process each message in the inbox
  repeat message_count_inbox [
    let message receive_first_message

    ; Extract relevant details from the message
    let building_patch item 0 message
    let owning_agent item 1 message

    ; Handle a demolition request
    if last message = "demolition_request" [
      if show_actions [show "Demolition company - Handle demolition request"]

      ; Check if the agent already has a link
      if member? owning_agent link-neighbors [
        set multiple_links_owners lput owning_agent multiple_links_owners
      ]
      ; Create a link to represent the ownership
      create-link-with owning_agent [
        set shape "dashed"
        set thickness 0.15
      ]

      ; Initialize the demolition project with necessary details
      let demolition_time 2
      let demolition_project (list building_patch owning_agent demolition_time)
      set projects lput demolition_project projects
    ]

    ; Handle response related to secondary materials
    if last message = "secondary_material_reponse" [
      if show_actions [show "Demolition company - Handle secondary material reponse"]

      ; Extract material details from the message
      let concrete_to_material_supplier item 1 message
      let wood_to_material_supplier item 2 message
      let steel_to_material_supplier item 3 message


      ; Adjust stock for each material
      set concrete_stock concrete_stock - concrete_to_material_supplier
      set wood_stock wood_stock - wood_to_material_supplier
      set steel_stock steel_stock - steel_to_material_supplier

      ; Update landfill counts
      set concrete_landfilled concrete_landfilled + concrete_stock
      set wood_landfilled wood_landfilled + wood_stock
      set steel_landfilled steel_landfilled + steel_stock

      ; Reset material stock
      set concrete_stock 0
      set wood_stock 0
      set steel_stock 0

    ]
  ]
end

; Checks and updates the status of active demolition projects
to check_demolition_projects
  let projects_checked []
  ; Exit if it's not a new month
  if not new_month [stop]

  ; Iterate over each project
  foreach projects [
    project ->

    ; Extract relevant details from the project
    let building_patch_func item 0 project
    let owning_agent_func item 1 project
    let remaining_demolition_time item 2 project

    ; If demolition is complete
    ifelse remaining_demolition_time = 0 [
      demolish_building building_patch_func
      if show_actions [show "Demolition company - Handle building demolition"]

      ; Handle agent links
      ifelse (position owning_agent_func multiple_links_owners) = FALSE [
        ask link-with owning_agent_func [die]
      ][
       set multiple_links_owners remove (position owning_agent_func multiple_links_owners) multiple_links_owners
      ]
    ]
    ; If demolition is still ongoing
    [
      let remaining_demolition_time_updated remaining_demolition_time - 1
      let project_updated (list building_patch_func owning_agent_func remaining_demolition_time_updated)
      set projects_checked lput project_updated projects_checked
    ]
  ]
  set projects projects_checked
end

; Resets the patch and handles the material flows after a building is demolished
to demolish_building[building_patch]
  ; Initialize extracted material counts
  let concrete_extracted 0
  let wood_extracted 0
  let steel_extracted 0

  ; Calculate extracted material based on the recycling rate
  ask building_patch [

    set concrete_demolished concrete_demolished + kg_concrete

    set wood_demolished wood_demolished + kg_wood
    set steel_demolished steel_demolished + kg_steel

    set concrete_extracted kg_concrete * collection_rate_concrete
    set wood_extracted kg_wood * collection_rate_wood
    set steel_extracted kg_steel * collection_rate_steel

    set concrete_landfilled concrete_landfilled + kg_concrete * (1 - collection_rate_concrete)
    set wood_landfilled wood_landfilled + kg_wood * (1 - collection_rate_wood)
    set steel_landfilled steel_landfilled + kg_steel * (1 - collection_rate_steel)
  ]

  ; Adjust material stock with extracted materials
  set concrete_stock concrete_stock + concrete_extracted
  set wood_stock wood_stock + wood_extracted
  set steel_stock steel_stock + steel_extracted

  let current_year read-from-string time:show dt "yyyy"

  ; Reset patch details after demolition
  ask building_patch [
    set status "empty"
    set pcolor white
    set kg_concrete 0
    set kg_wood 0
    set kg_steel 0
    set building_archetype "NULL"
    set building_construction_year current_year
    set building_floor_surface 0
    set remaining_lifespan 0
    set material_type "NULL"
    set time_empty 0
  ]

  ; Increase the count of demolished buildings
  set building_demolition_count building_demolition_count + 1
end


; This function clears demolition waste stocks if their total is non-positive.
to clear_demolition_waste_stock
  if not new_month [stop]
  ; If total waste stock is less than or equal to zero, stop the function.
  if (concrete_stock + wood_stock + steel_stock) <= 0 [stop]

  ; Create a request message for secondary material.
  let secondary_material_request_message (list material_supplier 7 self concrete_stock wood_stock steel_stock "secondary_material_request")

  ; Add the message to the outbox.
  set outbox lput secondary_material_request_message outbox
end

;--------------------------
; Owners
;--------------------------

; This function handles incoming messages for building owners.
to handle_inbox_owners
  ; If the inbox is empty, stop the function.
  if empty? inbox [stop]

  ; Initialize lists for project evaluation and acceptance.
  let project_evaluation_list []
  let project_acceptance_list []

  ; Count the number of messages in the inbox.
  let message_count_inbox length inbox
  repeat message_count_inbox [
    let message receive_first_message

    ; Check if the message type is "construction_cost_estimate".
    if last message = "construction_cost_estimate" [
      if show_actions [show "owners - construction cost estimate received"]
      ; Add the message (minus its last item) to the evaluation list.
      set project_evaluation_list lput but-last message project_evaluation_list
    ]
  ]

  ; If there are projects to evaluate, evaluate and update the acceptance list.
  if not empty? project_evaluation_list [
    set project_acceptance_list project_evaluation project_evaluation_list
  ]

  ; If there are accepted projects, send commission messages for each project.
  if not empty? project_acceptance_list [
    foreach project_acceptance_list [
      accepted_project ->
      let building_patch_func item 0 accepted_project
      let construction_agent_func item 1 accepted_project
      ; Extract other properties of the project.
      let material_type_func item 2 accepted_project
      let building_archetype_func item 3 accepted_project
      let building_floor_surface_func item 4 accepted_project

      ; Create and send a construction commission message.
      let construction_commission_message (list construction_agent_func building_patch_func self material_type_func building_archetype_func building_floor_surface_func "construction_commission")
      set outbox lput construction_commission_message outbox
    ]
  ]
end

; This function evaluates multiple construction projects and returns a list of the preferred ones.
to-report project_evaluation[project_evaluation_list]
  let project_acceptance_list []
  let patches_list map [project -> first project] project_evaluation_list
  ; Remove duplicate patches from the list.
  let unique_patches_list remove-duplicates patches_list

  ; Evaluate the construction cost estimates by patch and update the acceptance list.
  foreach unique_patches_list[
    patch_identifier ->
    let construction_cost_estimate_by_patch filter [project -> first project = patch_identifier] project_evaluation_list
    set project_acceptance_list lput (construction_project_decision_function construction_cost_estimate_by_patch) project_acceptance_list
  ]
  ; Return the list of accepted projects.
  report project_acceptance_list
end

; This function evaluates an owner's building assets and decides whether to request demolition or construction.
to evaluate_building_stock
  if not new_month [stop]
  let owning_agent self
  foreach assets [
    asset ->
    ask asset [
      ; If a building's lifespan has ended, request its demolition.
      if status = "occupied" and remaining_lifespan = 0 [
        set status "waiting_for_demolition_company"
        set pcolor brown
        let one_of_demolition_companies one-of demolition_companies
        let request (list one_of_demolition_companies asset owning_agent "demolition_request")
        ask owning_agent [set outbox lput request outbox]
      ]

      ; If a patch is empty, request a construction cost estimate.
      if status = "empty" [
        let potential_building_archetype first rnd:weighted-one-of-list distribution_build [ [p] -> last p ]
        let potential_building_floor_surface item 2 last dist_year_surf_single
        if building_archetype = "app" [set potential_building_floor_surface item 2 last dist_year_surf_multi]
        update_remaining_project_capacity

        ; Check if any construction company has the capacity to take on the project.
        ifelse check_maximum_capacity_construction_companies > potential_building_floor_surface [
          set status "waiting_for_construction_cost_estimate"
          set pcolor orange
          let all_construction_companies sort turtles with [breed = construction_companies]
          foreach all_construction_companies [
            construction_company_id ->
            let construction_cost_request (list construction_company_id asset owning_agent potential_building_archetype potential_building_floor_surface "construction_cost_request")
            ask owning_agent [set outbox lput construction_cost_request outbox]
          ]
        ][
          set status "empty"
          set pcolor black
        ]
      ]
    ]
  ]
end

; This function checks the maximum construction capacity among all construction companies and returns it.
to-report check_maximum_capacity_construction_companies
  let capacity_list []
  ; Collect the remaining project capacities of all construction companies.
  ask construction_companies [
    set capacity_list lput remaining_project_capacity capacity_list
  ]

  ; Return the highest capacity found.
  report max capacity_list
end



to-report construction_project_decision_function[construction_cost_estimate_by_patch]
  ;This function decides on which project to accept based on cost estimates and other preferences.

  ;Get the patch's construction project.
  let patch_construction_project item 0 item 0 construction_cost_estimate_by_patch

  ;Filter cost estimates by material type.
  let construction_cost_estimates_by_patch_wood_type filter [project -> item 2 project = "wood_type"] construction_cost_estimate_by_patch
  let construction_cost_estimates_by_patch_masonry_type filter [project -> item 2 project = "masonry_type"] construction_cost_estimate_by_patch

  ;Retrieve cost values for wood type.
  let construction_cost_values_wood_type map[project -> last project] construction_cost_estimates_by_patch_wood_type
  let lowest_cost_list_wood_type min construction_cost_values_wood_type
  let lowest_cost_list_postition_wood_type position lowest_cost_list_wood_type construction_cost_values_wood_type
  let lowest_cost_project_wood_type item lowest_cost_list_postition_wood_type construction_cost_estimates_by_patch_wood_type


  ;Retrieve cost values for masonry type.
  let construction_cost_values_masonry_type map[project -> last project] construction_cost_estimates_by_patch_masonry_type
  let lowest_cost_list_masonry_type min construction_cost_values_masonry_type
  let lowest_cost_list_postition_masonry_type position lowest_cost_list_masonry_type construction_cost_values_masonry_type
  let lowest_cost_project_masonry_type item lowest_cost_list_postition_masonry_type construction_cost_estimates_by_patch_masonry_type

  let wood_based_ratio 0
  let masonry_based_ratio 0

  let preceived_construction_cost_wood_type 0
  let preveived_construction_cost_masonry_type 0

  if breed != "housing_corporation_owner" [

    let neighborhood_wood_count 0
    let neighborhood_masonry_count 0


    ask patch_construction_project [
      ; Create a local variable to hold the patches in the extended neighborhood
      let extended_neighbors patches in-radius familiarity_radius_owners

      ; Count neighboring patches based on material type within the extended neighborhood.
      ; Note: We're using 'extended_neighbors' instead of 'neighbors' and we're excluding the center patch itself.
      set neighborhood_wood_count count (extended_neighbors with [material_type = "wood_type" and status = "occupied" and self != myself])
      set neighborhood_masonry_count count (extended_neighbors with [material_type = "masonry_type" and status = "occupied" and self != myself])
    ]



    let neighborhood_building_count neighborhood_wood_count + neighborhood_masonry_count

    carefully [set wood_based_ratio neighborhood_wood_count / neighborhood_building_count][set wood_based_ratio 0]
    carefully [set masonry_based_ratio neighborhood_masonry_count / neighborhood_building_count][set masonry_based_ratio 0]


    set preceived_construction_cost_wood_type last lowest_cost_project_wood_type * (1 - preference_bias * wood_based_ratio)
    set preveived_construction_cost_masonry_type last lowest_cost_project_masonry_type * (1 - preference_bias * masonry_based_ratio)

  ]

  ;For housing corporation owners, define preference based on existing assets.
  if breed = "housing_corporation_owner" [
    let wood_based_assets count assets with [material_type = "wood_type"]
    let masonry_based_assets count assets with [material_type = "masonry_type"]
    let total_asset_count wood_based_assets +  masonry_based_assets

    ; Calculate ratio's
    carefully [set wood_based_ratio wood_based_assets / total_asset_count][set wood_based_ratio 0]
    carefully [set masonry_based_ratio masonry_based_assets / total_asset_count][set masonry_based_ratio 0]

    ; Calculate preceived construction cost
    set preceived_construction_cost_wood_type last lowest_cost_project_wood_type * (1 - preference_bias * wood_based_ratio)
    set preveived_construction_cost_masonry_type last lowest_cost_project_masonry_type * (1 - preference_bias * masonry_based_ratio)

  ]



  ;Return the project with the highest cost.
  ifelse preveived_construction_cost_masonry_type < preceived_construction_cost_wood_type [

    report lowest_cost_project_masonry_type
  ][
    report lowest_cost_project_wood_type
  ]

end



;--------------------------
; Material Suppliers
;--------------------------

; This procedure handles incoming messages in the inbox of material suppliers.
to handle_inbox_material_suppliers

  ; Stop the procedure if there are no messages in the inbox.
  if empty? inbox [stop]

  ; Count the number of messages in the inbox.
  let message_count_inbox length inbox

  ; Process each message in the inbox.
  repeat message_count_inbox [

    ; Retrieve and store the first message from the inbox.
    let message receive_first_message

    ; Check if the last item in the message indicates a primary material request.
    if last message = "primary_material_cost_request" [
      ; Display an action if show_actions is true.
      if show_actions [show "Material supplier - primary material cost request"]

      ; Extract necessary data from the message.
      let building_patch_func item 0 message
      let construction_agent_func item 1 message
      let owning_agent_func item 2 message
      let building_archetype_func item 3 message
      let building_floor_surface_func item 4 message
      let material_type_func item 5 message
      let quantity_concrete_func item 6 message
      let quantity_wood_func item 7 message
      let quantity_steel_func item 8 message


      ; Calculate the material cost based on quantities.
      let material_cost calculate_material_cost quantity_concrete_func quantity_wood_func quantity_steel_func

      ; Create a response message with necessary details and the calculated material cost.
      let primary_material_response_message (list construction_agent_func building_patch_func self owning_agent_func building_archetype_func building_floor_surface_func material_type_func material_cost "primary_material_response")

      ; Add the response message to the outbox.
      set outbox lput primary_material_response_message outbox
    ]

    if last message = "primary_material_commission" [
      ; Display an action if show_actions is true.
      if show_actions [show "Material supplier - primary material commission"]

      ; Extract necessary data from the message.
      let construction_agent_func item 0 message
      let quantity_concrete_func item 1 message
      let quantity_wood_func item 2 message
      let quantity_steel_func item 3 message


      process_primary_material_request quantity_concrete_func concrete_stock "concrete"
      process_primary_material_request quantity_wood_func wood_stock "wood"
      process_primary_material_request quantity_steel_func steel_stock "steel"

    ]


    ; Check if the last item in the message indicates a secondary material request.
    if last message = "secondary_material_request" [

      ; Display an action if show_actions is true
      if show_actions [show "Material supplier - secondary material request"]

      ; Extract necessary data from the message
      let demolition_agent_func item 0 message
      let quantity_concrete_func item 1 message
      let quantity_wood_func item 2 message
      let quantity_steel_func item 3 message


      ; Process material requests and update stocks
      let concrete_quantity_acceptance process_secondary_material_request quantity_concrete_func capacity_concrete_stock_material_supplier concrete_stock "concrete"
      let wood_quantity_acceptance process_secondary_material_request quantity_wood_func capacity_wood_stock_material_supplier wood_stock "wood"
      let steel_quantity_acceptance process_secondary_material_request quantity_steel_func capacity_steel_stock_material_supplier steel_stock "steel"

      ; Create a response message for secondary material request
      let secondary_material_response_message (list demolition_agent_func self concrete_quantity_acceptance wood_quantity_acceptance steel_quantity_acceptance "secondary_material_reponse")

      ; Add the response message to the outbox
      set outbox lput secondary_material_response_message outbox
    ]
  ]
end

; Define a reporter procedure that processes secondary material requests
to-report process_secondary_material_request [requested_quantity capacity stock material_type_func]
  ; Initialize variables
  let accepted_quantity 0
  let requested_quantity_after_recyling 0
  let remaining_capacity capacity - stock
  let updated_stock 0
  let material_landfilled 0

  ; Calculate the quantity of material after considering recycling, based on material type
  ifelse material_type_func = "concrete" [
    set requested_quantity_after_recyling requested_quantity * recycling_rate_concrete
  ][ ifelse material_type_func = "wood" [
    set requested_quantity_after_recyling requested_quantity * recycling_rate_wood
  ][
    set requested_quantity_after_recyling requested_quantity * recycling_rate_steel
  ]]

  ; Determine the accepted quantity and update the stock based on remaining capacity
  if remaining_capacity > 0 [
    ifelse remaining_capacity >= requested_quantity_after_recyling [
      ; If remaining capacity is sufficient, accept the full requested quantity
      set accepted_quantity requested_quantity
      set updated_stock stock + requested_quantity_after_recyling
    ] [
      ; If not, accept only as much as the remaining capacity allows
      set accepted_quantity remaining_capacity
      set updated_stock capacity
    ]
  ]

  ; Update global variables for stock and landfill amounts based on material type
  ifelse material_type_func = "concrete" [
    set concrete_stock updated_stock
    set concrete_landfilled concrete_landfilled + accepted_quantity * (1 - recycling_rate_concrete)
  ][ ifelse material_type_func = "wood" [
    set wood_stock updated_stock
    set wood_landfilled wood_landfilled + accepted_quantity * (1 - recycling_rate_wood)
  ][
    set steel_stock updated_stock
    set steel_landfilled steel_landfilled +  accepted_quantity * (1 - recycling_rate_steel)
  ]]

  ; Report the accepted quantity of material
  report accepted_quantity
end




; This is the definition of the procedure to process primary material requests
to process_primary_material_request [requested_quantity stock material_type_func]

  ; Initialize the variable 'reduced_quantity' to keep track of the actual amount of material that can be provided
  let reduced_quantity 0
  ; Initialize the variable 'required_primary_material' to keep track of the additional material required if stock is insufficient
  let required_primary_material 0

  ; Check if the stock is greater than 0
  ifelse stock > 0 [
    ; If stock is enough to meet the requested quantity
    ifelse stock >= requested_quantity [
      ; Set the reduced quantity to the requested amount
      set reduced_quantity requested_quantity
    ] [
      ; If stock is not enough, set reduced quantity to stock and calculate the additional required material
      set reduced_quantity stock
      set required_primary_material requested_quantity - stock
    ]
  ][
    ; If no stock is available, all requested material is required
    set required_primary_material requested_quantity
  ]

  ; Update the global variables for stock, requested, and primary material based on the material type
  ifelse material_type_func = "concrete" [
    ; Update the concrete stock, requested, and primary material
    set concrete_stock concrete_stock - reduced_quantity
    set concrete_requested concrete_requested + requested_quantity
    set primary_concrete primary_concrete + required_primary_material
  ][ ifelse material_type_func = "wood" [
    ; Update the wood stock, requested, and primary material
    set wood_stock wood_stock - reduced_quantity
    set wood_requested wood_requested + requested_quantity
    set primary_wood primary_wood + required_primary_material
  ][
    ; Update the steel stock, requested, and primary material
    set steel_stock steel_stock - reduced_quantity
    set steel_requested steel_requested + requested_quantity
    set primary_steel primary_steel + required_primary_material
  ]]

end



; This function calculates and returns the total material cost based on the quantities and respective cost indices for concrete, wood, and steel.
to-report calculate_material_cost [quantity_concrete_func quantity_wood_func quantity_steel_func]



  ; Multiply each quantity with its respective cost index and sum them up to get the total material cost
  report
           quantity_wood_func * material_cost_wood
  + quantity_concrete_func * material_cost_concrete
          + quantity_steel_func * material_cost_steel
end

;----------------------------------------------------------------------------------------------------------------------

; Messaging system section

; This function retrieves the first message from the agent's inbox and removes it from the inbox.
to-report receive_first_message

  ; Get the first message from the inbox
  let first_message item 0 inbox

  ; Remove the first message from the inbox
  set inbox remove-item 0 inbox

  ; Return the first message
  report first_message
end

; This function sends all the messages present in the outbox of the sender agent to the inbox of the receiver agent.
to send_messages_outbox

  ; If outbox is empty, exit the procedure
  if empty? outbox [stop]

  ; Display the outbox messages if the show_messages flag is ON
  if show_messages [show outbox]

  ; Display the action if the show_actions flag is ON
  if show_actions [show "sending messages outbox"]

  ; Count the number of messages in the outbox
  let message_count_outbox length outbox

  ; Loop through each message in the outbox
  repeat message_count_outbox [

    ; Fetch the first message from the outbox
    let first_message item 0 outbox

    ; Send the message to the receiver's inbox. 'item 0 first_message' is the agent who will receive the message.
    ask item 0 first_message [
      ; Add the message (without the receiver agent's name) to the receiver's inbox
      set inbox lput (but-first first_message) inbox
    ]

    ; Remove the first message from the sender's outbox
    set outbox remove-item 0 outbox
  ]

  ; Clear the outbox
  set outbox []
end
@#$#@#$#@
GRAPHICS-WINDOW
13
123
641
752
-1
-1
15.122
1
20
1
1
1
0
0
0
1
0
40
0
40
0
0
1
Months
30.0

BUTTON
6
10
69
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
80
10
152
43
Run
while [ticks < tick_count_input] [ go ]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
916
491
1063
536
Demolition counter
building_demolition_count
17
1
11

MONITOR
62
51
115
96
Simulation to real ratio
ams_building_count_starting_year / count patches with [part_of_built_environment = 1]
0
1
11

MONITOR
915
446
1063
491
Construction counter
building_construction_count
17
1
11

MONITOR
122
52
172
97
Links
count links
17
1
11

SWITCH
388
10
535
43
show_messages
show_messages
1
1
-1000

SWITCH
388
48
536
81
show_actions
show_actions
1
1
-1000

MONITOR
699
445
878
490
Specialization - CC0
[specialization] of construction_company 0
2
1
11

MONITOR
699
493
879
538
Specialization - CC1
[specialization] of construction_company 1
2
1
11

MONITOR
700
539
880
584
Specialization - CC2
[specialization] of construction_company 2
2
1
11

MONITOR
1063
491
1196
536
Wood based buildings
count patches with [material_type = \"wood_type\"]
17
1
11

MONITOR
6
51
56
96
ticks
ticks
17
1
11

BUTTON
162
10
245
43
Run once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1062
446
1194
491
Masonry type buildings
count patches with [material_type = \"masonry_type\"]
17
1
11

SWITCH
553
12
693
45
show_new_tick
show_new_tick
1
1
-1000

INPUTBOX
238
50
354
110
tick_count_input
600.0
1
0
Number

BUTTON
252
10
353
43
Debug mode
ask patches with [status = \"occupied\"]  [set remaining_lifespan 1000000]\nask patch 7 7 [set remaining_lifespan 0]\nask patch 7 7 [set building_floor_surface 10]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
554
53
716
86
sensitivity_analysis
sensitivity_analysis
0
1
-1000

PLOT
758
20
1205
354
Ratio wood-based buildings
Time (ticks)
Ratio
0.0
1.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot ratio_wood_based_buildings"

PLOT
1239
20
1797
355
primary_[material] / [material]_requested
Time (ticks)
Ratio
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Ratio concrete" 1.0 0 -16777216 true "" "plot ratio_concrete_primary_requested"
"Ration wood" 1.0 0 -7500403 true "" "plot ratio_wood_primary_requested"
"Ratio steel" 1.0 0 -2674135 true "" "plot ratio_steel_primary_requested"

PLOT
1240
383
1802
722
[material]_landfilled / [material]_demolished
Time (ticks)
Ratio
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"Ratio concrete" 1.0 0 -16777216 true "" "plot ratio_concrete_landfilled_demolished"
"Ratio wood" 1.0 0 -7500403 true "" "plot ratio_wood_landfilled_demolished"
"Ratio steel" 1.0 0 -2674135 true "" "plot ratio_steel_landfilled_demolished"

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

garbage can
false
0
Polygon -16777216 false false 60 240 66 257 90 285 134 299 164 299 209 284 234 259 240 240
Rectangle -7500403 true true 60 75 240 240
Polygon -7500403 true true 60 238 66 256 90 283 135 298 165 298 210 283 235 256 240 238
Polygon -7500403 true true 60 75 66 57 90 30 135 15 165 15 210 30 235 57 240 75
Polygon -7500403 true true 60 75 66 93 90 120 135 135 165 135 210 120 235 93 240 75
Polygon -16777216 false false 59 75 66 57 89 30 134 15 164 15 209 30 234 56 239 75 235 91 209 120 164 135 134 135 89 120 64 90
Line -16777216 false 210 120 210 285
Line -16777216 false 90 120 90 285
Line -16777216 false 125 131 125 296
Line -16777216 false 65 93 65 258
Line -16777216 false 175 131 175 296
Line -16777216 false 235 93 235 258
Polygon -16777216 false false 112 52 112 66 127 51 162 64 170 87 185 85 192 71 180 54 155 39 127 36

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

i beam
false
0
Polygon -7500403 true true 165 15 240 15 240 45 195 75 195 240 240 255 240 285 165 285
Polygon -7500403 true true 135 15 60 15 60 45 105 75 105 240 60 255 60 285 135 285

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

logs
false
0
Polygon -7500403 true true 15 241 75 271 89 245 135 271 150 246 195 271 285 121 235 96 255 61 195 31 181 55 135 31 45 181 49 183
Circle -1 true false 132 222 66
Circle -16777216 false false 132 222 66
Circle -1 true false 72 222 66
Circle -1 true false 102 162 66
Circle -7500403 true true 222 72 66
Circle -7500403 true true 192 12 66
Circle -7500403 true true 132 12 66
Circle -16777216 false false 102 162 66
Circle -16777216 false false 72 222 66
Circle -1 true false 12 222 66
Circle -16777216 false false 30 240 30
Circle -1 true false 42 162 66
Circle -16777216 false false 42 162 66
Line -16777216 false 195 30 105 180
Line -16777216 false 255 60 165 210
Circle -16777216 false false 12 222 66
Circle -16777216 false false 90 240 30
Circle -16777216 false false 150 240 30
Circle -16777216 false false 120 180 30
Circle -16777216 false false 60 180 30
Line -16777216 false 195 270 285 120
Line -16777216 false 15 240 45 180
Line -16777216 false 45 180 135 30

orbit 5
true
0
Circle -7500403 true true 116 11 67
Circle -7500403 true true 13 89 67
Circle -7500403 true true 178 206 67
Circle -7500403 true true 53 204 67
Circle -7500403 true true 220 91 67
Circle -7500403 false true 45 45 210

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.3.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

dashed
0.0
-0.2 0 0.0 1.0
0.0 1 4.0 4.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
